//
// GXSample.cpp : Pocket PC Game API Starfield sample.
//
// Mostly C in a C++ file.
// See the documentation for more information.


//
// Includes
//

#define WIN32_LEAN_AND_MEAN		// Exclude rarely-used stuff from Windows headers
#include <windows.h>
#include "gx.h"

// Defines

#define kcxStarField 512		// width and height of star field (power of 2)
#define kcyStarField 512
#define kmaskStarField (kcxStarField - 1)
#define dmaxVelocity 100
#define kscSpeed 50				// speed scale
#define kcStars 2000

// Definitions

struct STAR {
	unsigned short x;
	unsigned short y;
};

//
// Some handy globals that are fine for a sample app.
//

STAR astStars[kcStars];			// array of star values
GXDisplayProperties g_gxdp;		// GX struct
GXKeyList g_gxkl;				// GX struct
int g_dxVelocity, g_dyVelocity;	// velocity vector
int g_dxThrust, g_dyThrust;		// thrust vector
int g_dxErr, g_dyErr;			// error vector
bool g_fDrag;					// in drag mode or not
int g_xDrag, g_yDrag;			// for calculating vector in drag mode
bool g_fPause = false;			// paused or not

//
// Forward declarations.
//

ATOM MyRegisterClass(HINSTANCE hInstance, LPTSTR szWindowClass);
BOOL InitInstance(HINSTANCE hInstance, int nCmdShow);
LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
bool ClearScreen(int iParam);
void InitStars();
void DrawStars(int dx, int dy);

//
// Code Body
//

int WINAPI WinMain(	HINSTANCE hInstance,
					HINSTANCE hPrevInstance,
					LPTSTR    lpCmdLine,
					int       nCmdShow)
{
	MSG msg;

	// Perform application initialization:

	if (!InitInstance (hInstance, nCmdShow)) {
		return FALSE;
	}

	// Main message loop.

	while (GetMessage(&msg, NULL, 0, 0)) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}


	return msg.wParam;
}

//
//  FUNCTION: MyRegisterClass()
//
//  PURPOSE: Registers the window class.
//

ATOM MyRegisterClass(HINSTANCE hInstance, LPTSTR szWindowClass)
{
	WNDCLASS	wc;

    wc.style			= CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc		= (WNDPROC) WndProc;
    wc.cbClsExtra		= 0;
    wc.cbWndExtra		= 0;
    wc.hInstance		= hInstance;
    wc.hIcon			= NULL;
    wc.hCursor			= 0;
    wc.hbrBackground	= (HBRUSH) GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName		= 0;
    wc.lpszClassName	= szWindowClass;

	return RegisterClass(&wc);
}

BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)
{
	HWND	hWnd;
	TCHAR	szTitle[] = L"GXSample";			// The title bar text
	TCHAR	szWindowClass[] = L"GXSample";	// The window class name

	// IMPORTANT:  see if this app is already running.  If so just bring
	// it to the front and quit.  All CE apps need to do this.

	hWnd = FindWindow(szWindowClass, szTitle);	
	if (hWnd) {
		SetForegroundWindow ((HWND) (((DWORD)hWnd) | 0x01));    
		return 0;
	} 

	MyRegisterClass(hInstance, szWindowClass);

	// In order to create a full screen app CreateWindow() needs to be
	// called with absolute coordinates that cover the entire display.
	// Using CW_USEDEFAULT will not work correctly.

	hWnd = CreateWindow(szWindowClass, szTitle, WS_VISIBLE,
		0, 0, GetSystemMetrics(SM_CXSCREEN), GetSystemMetrics(SM_CYSCREEN), NULL, NULL, hInstance, NULL);

	if (!hWnd) {	
		return FALSE;
	}

	ShowWindow(hWnd, nCmdShow);
	UpdateWindow(hWnd);

	if (GXOpenDisplay(hWnd, GX_FULLSCREEN) == 0) {
		return FALSE;				// we won't be able to draw.
	}

	WCHAR wszMachineName[128];
	SystemParametersInfo(SPI_GETOEMINFO, sizeof(wszMachineName), &wszMachineName, 0);
	if (wcsicmp(wszMachineName, L"HP,Jornada 540") == 0) {	// Jornada 540
		return TRUE;
	}

	GXOpenInput();

	g_gxdp = GXGetDisplayProperties();
	g_gxkl = GXGetDefaultKeys(GX_NORMALKEYS);

	return TRUE;
}

//
//  FUNCTION: WndProc(HWND, unsigned, WORD, LONG)
//
//  PURPOSE:  Processes messages for the main window.
//

LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HDC hdc;
	PAINTSTRUCT ps;
	short vkKey;

	switch (message) {
		case WM_CREATE:
			SetTimer(hWnd, 101, 10, NULL);			// A fast timer.
			InitStars();
			break;

		case WM_TIMER:

			// Add the thrust vector to the velocity vector.

			g_dxVelocity += g_dxThrust;
			g_dyVelocity += g_dyThrust;

			// Clamp the velocity vector.

			if (g_dxVelocity > dmaxVelocity)
				g_dxVelocity = dmaxVelocity;
			if (g_dxVelocity < -dmaxVelocity)
				g_dxVelocity = -dmaxVelocity;

			if (g_dyVelocity > dmaxVelocity)
				g_dyVelocity = dmaxVelocity;
			if (g_dyVelocity < -dmaxVelocity)
				g_dyVelocity = -dmaxVelocity;

			// Move stars by the velocity vector scaled down by the speed scale factor.

			DrawStars((g_dxVelocity + g_dxErr) / kscSpeed, (g_dyVelocity + g_dyErr) / kscSpeed);

			// Keep track of error:  scaling the velocity vector using integer math introduced
			// an inaccuracy.  Add that inaccuracy in next time we move the stars.

			g_dxErr = (g_dxVelocity + g_dxErr) % kscSpeed;
			g_dyErr = (g_dyVelocity + g_dyErr) % kscSpeed;
			break;

		case WM_ACTIVATE:
			break;

		case WM_KILLFOCUS:

			// Always, always call GXSuspend() here.
			
			GXSuspend();
			break;


		case WM_SETFOCUS:

			// And always call GXResume() here.

			GXResume();
			break;

		// If your app can only draw directly to the screen then you need to make
		// sure it is frontmost before drawing.

		case WM_PAINT:
			hdc = BeginPaint(hWnd, &ps);
			if (GetForegroundWindow() == hWnd) {
				ClearScreen(0);
				DrawStars(0, 0);
			}
			EndPaint(hWnd, &ps);
			break;

		case WM_KEYDOWN:
			vkKey = (short)wParam;
			if (vkKey == g_gxkl.vkUp) {
				g_dyThrust = -1;
				break;
			}

			if (vkKey == g_gxkl.vkDown) {
				g_dyThrust = 1;
				break;
			}

			if (vkKey == g_gxkl.vkLeft) {
				g_dxThrust = -1;
				break;
			}

			if (vkKey == g_gxkl.vkRight) {
				g_dxThrust = 1;
				break;
			}

			if (vkKey == g_gxkl.vkA) {
				ClearScreen(1);
				break;
			}

			if (vkKey == g_gxkl.vkB) {
				if (g_fPause == true) {
					SetTimer(hWnd, 101, 10, NULL);			// A fast timer.
					g_fPause = false;
				} else {
					g_fPause = true;
					KillTimer(hWnd, 101);
				}

				break;
			}

			if (vkKey == g_gxkl.vkC) {
				ClearScreen(1);						// Clear to random color
				break;
			}

			if (vkKey == g_gxkl.vkStart) {
				SendMessage(hWnd, WM_CLOSE, 0, 0);
				break;
			}


			break;

		case WM_KEYUP:
			vkKey = (short)wParam;
			if (vkKey == g_gxkl.vkUp) {
				g_dyThrust = 0;
				break;
			}

			if (vkKey == g_gxkl.vkDown) {
				g_dyThrust = 0;
				break;
			}

			if (vkKey == g_gxkl.vkLeft) {
				g_dxThrust = 0;
				break;
			}

			if (vkKey == g_gxkl.vkRight) {
				g_dxThrust = 0;
				break;
			}
			break;

		case WM_LBUTTONDOWN:

			// Allow the user to grab the starfield and drag it around.

			KillTimer(hWnd, 101);
			g_dxVelocity = 0;
			g_dyVelocity = 0;
			g_dxThrust = 0;
			g_dyThrust = 0;
			g_dxErr = 0;
			g_dyErr = 0;

			g_fDrag = true;
			g_xDrag = LOWORD(lParam);
			g_yDrag = HIWORD(lParam);
			break;

		case WM_LBUTTONUP:

			// Turn the timer back on.

			g_fDrag = false;
			SetTimer(hWnd, 101, 10, NULL);			// A fast timer.
			break;

		case WM_MOUSEMOVE:
 			if (g_fDrag == true) {
				DrawStars(LOWORD(lParam) - g_xDrag, HIWORD(lParam) - g_yDrag);
				g_xDrag = LOWORD(lParam);
				g_yDrag = HIWORD(lParam);
			}
			break;

		case WM_DESTROY:
			GXCloseInput();			// for maximum compatibility with certain HP hardware, do this here or better yet in WM_CLOSE.
			GXCloseDisplay();		// this can go just about anywhere, but since input is being closed, may as well do it here.
			PostQuitMessage(0);
			break;

		default:
			return DefWindowProc(hWnd, message, wParam, lParam);
	}

	return 0;
}

//
// 0 clears the screen to black.
// any other value clears the screen to a random value.
//

bool ClearScreen(int iParam)
{
	if (iParam != 0) {
		iParam = (Random() & 0xff);
	}

	// 16 bit per pixel code.  Note 2 different pixel formats.

	if (g_gxdp.cBPP == 16) {
		unsigned short usPixel = 0;
		if (g_gxdp.ffFormat | kfDirect565) {
			usPixel = (unsigned short) (((Random() & 0xff) >> 3) << 11 | ((Random() & 0xff) >> 2) << 5 | ((Random() & 0xff) >> 3));
		} else if (g_gxdp.ffFormat | kfDirect555) {
			usPixel = (unsigned short) (((Random() & 0xff) >> 3) << 10 | ((Random() & 0xff) >> 3) << 5 | ((Random() & 0xff) >> 3));
		}

		if (iParam == 0) {
			usPixel = 0;
		}

		unsigned short * pusLine = (unsigned short *)GXBeginDraw();
		if (pusLine == NULL)
			return false;			// NOT OK TO DRAW, return failure.

		for (unsigned int y = 0; y < g_gxdp.cyHeight; y++) {
			unsigned short * pusDest = pusLine;
			for (unsigned int x = 0; x < g_gxdp.cxWidth; x++) {
				*pusDest = usPixel;
				pusDest += g_gxdp.cbxPitch >> 1;
			}
			pusLine += g_gxdp.cbyPitch >> 1;
		}

		GXEndDraw();

		return true;
	}

	// 8 bit per pixel code.

	if (g_gxdp.cBPP == 8) {
		unsigned char bPixel = iParam & 0xff;

		unsigned char * pbLine = (unsigned char *)GXBeginDraw();
		if (pbLine == NULL)
			return false;			// NOT OK TO DRAW, return failure.

		for (unsigned int y = 0; y < g_gxdp.cyHeight; y++) {
			unsigned char * pbDest = pbLine;
			for (unsigned int x = 0; x < g_gxdp.cxWidth; x++) {
				*pbDest = bPixel;
				pbDest += g_gxdp.cbxPitch;
			}
			pbLine += g_gxdp.cbyPitch;
		}

		GXEndDraw();		// don't forget, or you'll be scratching your head...

		return true;
	}

	// 4 bit per pixel code.

	if (g_gxdp.cBPP == 4 && !(g_gxdp.ffFormat & kfLandscape)) {
		unsigned char bPixel = iParam & 0xff;

		unsigned char * pbLine = (unsigned char *)GXBeginDraw();
		if (pbLine == NULL)
			return false;			// NOT OK TO DRAW, return failure.

		for (unsigned int y = 0; y < g_gxdp.cyHeight; y++) {
			unsigned char * pbDest = pbLine;
			for (unsigned int x = 0; x < g_gxdp.cxWidth >> 1; x++) {
				*pbDest++ = bPixel;
			}
			pbLine += g_gxdp.cbyPitch;
		}

		GXEndDraw();		// don't forget, or you'll be scratching your head...

		return true;
	}

	// 2 bit per pixel code is left as an exercise to the reader.

	return false;
}

//
// Populates the starfield with random stars.
//

void InitStars()
{
	for (int i = 0; i < kcStars; i++) {
		astStars[i].x = (unsigned short) Random() & kmaskStarField;
		astStars[i].y = (unsigned short) Random() & kmaskStarField;
	}
}

//
// Fairly optimal code for drawing the starfield.  Made it clearer rather than faster.
//

void DrawStars(int dx, int dy)
{
	// Generic BeginDraw

	void * pvDisplay = GXBeginDraw();
	if (pvDisplay == NULL) {
		return;
	}

	// 4 bit per pixel regular (non-landscape)

	if (g_gxdp.cBPP == 4 && !(g_gxdp.ffFormat & kfLandscape)) {
		unsigned char * pbDst = (unsigned char *)pvDisplay;
		for (int i = 0; i < kcStars; i++) {

			// Erase star if it's on the screen.

			if (astStars[i].x < (unsigned)g_gxdp.cxWidth && astStars[i].y < (unsigned)g_gxdp.cyHeight) {
				*(pbDst + (astStars[i].x >> 1) + g_gxdp.cbyPitch * astStars[i].y) = 0;
			}

			// Two fields of stars; one moves faster than the other.

			if (i & 1) {
				astStars[i].x = (astStars[i].x + dx) & kmaskStarField;
				astStars[i].y = (astStars[i].y + dy) & kmaskStarField;
			} else {
				astStars[i].x = (astStars[i].x + (dx << 1)) & kmaskStarField;;
				astStars[i].y = (astStars[i].y + (dy << 1)) & kmaskStarField;
			}

			// Draw star if it's on the screen.

			if (astStars[i].x < (unsigned) g_gxdp.cxWidth && astStars[i].y < (unsigned) g_gxdp.cyHeight) {
				*(pbDst + (astStars[i].x >> 1) + g_gxdp.cbyPitch * astStars[i].y) |= (astStars[i].x & 1) ? 0x0f : 0xf0;
			}
		}

	}

	// 8 bit per pixel code.

	if (g_gxdp.cBPP == 8) {
		unsigned char * pbDst = (unsigned char *)pvDisplay;
		for (int i = 0; i < kcStars; i++) {

			// Erase star if it's on the screen.

			if (astStars[i].x < (unsigned)g_gxdp.cxWidth && astStars[i].y < (unsigned)g_gxdp.cyHeight) {
				*(pbDst + g_gxdp.cbxPitch * astStars[i].x + g_gxdp.cbyPitch * astStars[i].y) = 0;
			}

			// Two fields of stars; one moves faster than the other.

			if (i & 1) {
				astStars[i].x = (astStars[i].x + dx) & kmaskStarField;
				astStars[i].y = (astStars[i].y + dy) & kmaskStarField;
			} else {
				astStars[i].x = (astStars[i].x + (dx << 1)) & kmaskStarField;;
				astStars[i].y = (astStars[i].y + (dy << 1)) & kmaskStarField;
			}

			// Draw star if it's on the screen.

			if (astStars[i].x < (unsigned) g_gxdp.cxWidth && astStars[i].y < (unsigned) g_gxdp.cyHeight) {
				*(pbDst + g_gxdp.cbxPitch * astStars[i].x + g_gxdp.cbyPitch * astStars[i].y) = 255;
			}
		}

	}

	// And the nearly identical 16 bits per pixel code.  This is a job for templates!
	// That too is an exercise for the reader.
	//
	// Notice how all the pointers are still byte pointers so that the pitch arithmetic
	// works, and then are cast to unsigned short pointer at the last moment.  Alternately
	// you could use short pointers and shift the pitch values down 1 before doing
	// any computation with them.

	if (g_gxdp.cBPP == 16) {
		unsigned char * pbDst = (unsigned char *)pvDisplay;
		for (int i = 0; i < kcStars; i++) {

			// Erase star if it's on the screen.

			if (astStars[i].x < (unsigned)g_gxdp.cxWidth && astStars[i].y < (unsigned)g_gxdp.cyHeight) {
				*(unsigned short *)(pbDst + g_gxdp.cbxPitch * astStars[i].x + g_gxdp.cbyPitch * astStars[i].y) = 0;
			}

			// Two fields of stars; one moves faster than the other.

			if (i & 1) {
				astStars[i].x = (astStars[i].x + dx) & kmaskStarField;
				astStars[i].y = (astStars[i].y + dy) & kmaskStarField;
			} else {
				astStars[i].x = (astStars[i].x + (dx << 1)) & kmaskStarField;;
				astStars[i].y = (astStars[i].y + (dy << 1)) & kmaskStarField;
			}

			// Draw star if it's on the screen.

			if (astStars[i].x < (unsigned) g_gxdp.cxWidth && astStars[i].y < (unsigned) g_gxdp.cyHeight) {
				*(unsigned short *)(pbDst + g_gxdp.cbxPitch * astStars[i].x + g_gxdp.cbyPitch * astStars[i].y) = 0xffff;
			}
		}

	}

	GXEndDraw();
}


